<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphOne;

class Payment extends Model
{
    protected $fillable = [
        'rental_id',
        'amount',
        'payment_date',
        'payment_type',   // 'DP' | 'Pelunasan' | 'Denda'
        'payment_method', // 'Cash' | 'Transfer' | 'E-Wallet'
        'notes',
        'proof_file',
    ];

    protected $casts = [
        'payment_date' => 'datetime',
        'amount'       => 'decimal:2',
    ];

    public function rental(): BelongsTo
    {
        return $this->belongsTo(Rental::class);
    }

    public function financialReport(): MorphOne
    {
        return $this->morphOne(FinancialReport::class, 'source');
    }

    protected static function booted(): void
    {
        static::created(function (Payment $payment) {
            $payment->financialReport()->create([
                'transaction_date' => $payment->payment_date,
                'type'             => 'credit', // pembayaran = uang masuk
                'amount'           => $payment->amount,
                'description'      => sprintf(
                    'Pembayaran %s untuk Rental #%s',
                    $payment->payment_type,
                    optional($payment->rental)->rental_number ?? $payment->rental_id
                ),
            ]);
        });

        static::updated(function (Payment $payment) {
            $payment->financialReport()->update([
                'transaction_date' => $payment->payment_date,
                'amount'           => $payment->amount,
                'description'      => sprintf(
                    'Pembayaran %s untuk Rental #%s',
                    $payment->payment_type,
                    optional($payment->rental)->rental_number ?? $payment->rental_id
                ),
            ]);
        });

        static::deleted(function (Payment $payment) {
            $payment->financialReport()->delete();
        });
    }
}
