<?php

namespace App\Filament\Resources;

use App\Filament\Resources\RentalResource\Pages;
use App\Filament\Resources\RentalResource\RelationManagers;
use App\Filament\Resources\RentalResource\RelationManagers\PaymentsRelationManager;
use App\Models\Rental;
use App\Models\Motorcycle;
use App\Models\Customer;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Carbon\Carbon;

class RentalResource extends Resource
{
    protected static ?string $model = Rental::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Rental Motor';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('customer_id')
                    ->label('Pelanggan')
                    ->relationship('customer', 'name')
                    ->required(),

                Forms\Components\Select::make('motorcycle_id')
                    ->label('Motor')
                    ->relationship('motorcycle', 'plate_number')
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $set) {
                        if ($motor = Motorcycle::find($state)) {
                            $set('price_per_day', $motor->daily_price);
                        }
                    })
                    ->required(),

                Forms\Components\DateTimePicker::make('start_date')
                    ->label('Tanggal Mulai')
                    ->required()
                    ->reactive()
                    ->afterStateUpdated(fn ($state, callable $set, callable $get) =>
                        self::updateTotalPrice($set, $get)
                    ),

                Forms\Components\DateTimePicker::make('end_date')
                    ->label('Tanggal Selesai')
                    ->required()
                    ->reactive()
                    ->afterStateUpdated(fn ($state, callable $set, callable $get) =>
                        self::updateTotalPrice($set, $get)
                    ),

                Forms\Components\TextInput::make('price_per_day')
                    ->label('Harga per Hari')
                    ->numeric()
                    ->required()
                    ->reactive()
                    ->afterStateUpdated(fn ($state, callable $set, callable $get) =>
                        self::updateTotalPrice($set, $get)
                    ),

                Forms\Components\TextInput::make('total_price')
                    ->label('Total Harga')
                    ->numeric()
                    ->required(),

                Forms\Components\Select::make('status')
                    ->label('Status')
                    ->options([
                        'booked' => 'Booked',
                        'ongoing' => 'Ongoing',
                        'completed' => 'Completed',
                        'late' => 'Late',
                        'cancelled' => 'Cancelled',
                    ])
                    ->default('booked'),

                Forms\Components\Textarea::make('notes')
                    ->label('Catatan')
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('rental_number')->label('No. Sewa')->searchable(),
                Tables\Columns\TextColumn::make('customer.name')->label('Pelanggan'),
                Tables\Columns\TextColumn::make('motorcycle.plate_number')->label('No. Polisi'),
                Tables\Columns\TextColumn::make('start_date')->dateTime(),
                Tables\Columns\TextColumn::make('end_date')->dateTime(),
                Tables\Columns\BadgeColumn::make('status')->colors([
                    'warning' => 'booked',
                    'success' => 'completed',
                    'danger' => 'late',
                    'primary' => 'ongoing',
                    'secondary' => 'cancelled',
                ]),
                Tables\Columns\TextColumn::make('total_price')->money('IDR'),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            PaymentsRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListRentals::route('/'),
            'create' => Pages\CreateRental::route('/create'),
            'edit' => Pages\EditRental::route('/{record}/edit'),
        ];
    }

    public static function updateTotalPrice($set, $get)
    {
        $start = $get('start_date');
        $end = $get('end_date');
        $price = $get('price_per_day');

        if ($start && $end && $price) {
            $days = Carbon::parse($start)->diffInDays(Carbon::parse($end)) + 1;
            $set('total_price', $days * $price);
        }
    }

    public static function mutateFormDataBeforeCreate(array $data): array
    {
        $data['rental_number'] = 'RENT-' . now()->format('Ymd') . '-' . str_pad((Rental::count() + 1), 3, '0', STR_PAD_LEFT);
        return $data;
    }
}
