<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PaymentResource\Pages;
use App\Filament\Resources\PaymentResource\RelationManagers;
use App\Models\Payment;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class PaymentResource extends Resource
{
    protected static ?string $model = Payment::class;

    protected static ?string $navigationIcon  = 'heroicon-o-banknotes';
    protected static ?string $navigationGroup = 'Keuangan';
    protected static ?string $modelLabel      = 'Pembayaran';
    protected static ?string $pluralModelLabel = 'Payments';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('rental_id')
                ->label('Rental')
                ->relationship('rental', 'rental_number')
                ->searchable()
                ->preload()
                ->required(),

            Forms\Components\TextInput::make('amount')
                ->label('Jumlah')
                ->numeric()
                ->prefix('Rp')
                ->required(),

            Forms\Components\Select::make('payment_type')
                ->label('Tipe Pembayaran')
                ->options([
                    'DP'        => 'DP',
                    'Pelunasan' => 'Pelunasan',
                    'Denda'     => 'Denda',
                ])
                ->required(),

            Forms\Components\Select::make('payment_method')
                ->label('Metode')
                ->options([
                    'Cash'     => 'Cash',
                    'Transfer' => 'Transfer',
                    'E-Wallet' => 'E-Wallet',
                ])
                ->required(),

            Forms\Components\DateTimePicker::make('payment_date')
                ->label('Tanggal & Waktu Bayar')
                ->seconds(false)
                ->required(),

            Forms\Components\Textarea::make('notes')
                ->label('Catatan')
                ->columnSpanFull(),
            ])->columns(2);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('rental.rental_number')
                    ->label('No. Sewa')
                    ->searchable(),

                Tables\Columns\TextColumn::make('rental.customer.name')
                    ->label('Pelanggan')
                    ->toggleable()
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('amount')
                    ->label('Jumlah')
                    ->money('IDR', true)
                    ->sortable(),

                Tables\Columns\BadgeColumn::make('payment_type')
                    ->label('Tipe')
                    ->colors([
                        'warning' => 'DP',
                        'success' => 'Pelunasan',
                        'danger'  => 'Denda',
                    ])
                    ->sortable(),

                Tables\Columns\BadgeColumn::make('payment_method')
                    ->label('Metode')
                    ->colors([
                        'primary' => 'Cash',
                        'info'    => 'Transfer',
                        'success' => 'E-Wallet',
                    ])
                    ->sortable(),

                Tables\Columns\TextColumn::make('payment_date')
                    ->label('Dibayar Pada')
                    ->dateTime('d M Y H:i')
                    ->sortable(),

                Tables\Columns\TextColumn::make('notes')
                    ->label('Catatan')
                    ->limit(30)
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('payment_type')
                    ->label('Tipe')
                    ->options([
                        'DP'        => 'DP',
                        'Pelunasan' => 'Pelunasan',
                        'Denda'     => 'Denda',
                    ]),
                Tables\Filters\SelectFilter::make('payment_method')
                    ->label('Metode')
                    ->options([
                        'Cash'     => 'Cash',
                        'Transfer' => 'Transfer',
                        'E-Wallet' => 'E-Wallet',
                    ]),
                Tables\Filters\Filter::make('tanggal')
                    ->form([
                        Forms\Components\DatePicker::make('from')->label('Dari'),
                        Forms\Components\DatePicker::make('until')->label('Sampai'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when($data['from'] ?? null, fn ($q, $from) => $q->whereDate('payment_date', '>=', $from))
                            ->when($data['until'] ?? null, fn ($q, $until) => $q->whereDate('payment_date', '<=', $until));
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->requiresConfirmation(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])->defaultSort('payment_date', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPayments::route('/'),
            'create' => Pages\CreatePayment::route('/create'),
            'edit' => Pages\EditPayment::route('/{record}/edit'),
        ];
    }
}
