<?php

namespace App\Filament\Resources;

use App\Filament\Resources\FinancialReportResource\Pages;
use App\Filament\Resources\FinancialReportResource\RelationManagers;
use App\Models\FinancialReport;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class FinancialReportResource extends Resource
{
    protected static ?string $model = FinancialReport::class;

    protected static ?string $navigationIcon  = 'heroicon-o-chart-bar';
    protected static ?string $navigationGroup = 'Keuangan';
    protected static ?string $modelLabel      = 'Laporan Keuangan';
    protected static ?string $pluralModelLabel = 'Financial Reports';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('transaction_date')
                    ->label('Tanggal')
                    ->dateTime('d M Y H:i')
                    ->sortable(),

                Tables\Columns\BadgeColumn::make('type')
                    ->label('Tipe')
                    ->colors([
                        'success' => 'credit', // uang masuk
                        'danger'  => 'debit',  // uang keluar
                    ])
                    ->formatStateUsing(fn ($state) => ucfirst($state))
                    ->sortable(),

                Tables\Columns\TextColumn::make('amount')
                    ->label('Jumlah')
                    ->money('IDR', true)
                    ->sortable(),

                Tables\Columns\TextColumn::make('description')
                    ->label('Deskripsi')
                    ->limit(40)
                    ->tooltip(fn ($record) => $record->description),

                Tables\Columns\TextColumn::make('source_type')
                    ->label('Sumber')
                    ->formatStateUsing(function ($state) {
                        return class_exists($state)
                            ? class_basename($state)
                            : $state;
                    })
                    ->toggleable(),

                Tables\Columns\TextColumn::make('source_id')
                    ->label('Ref ID')
                    ->toggleable(),

                // Tampilkan bukti transfer bila sumbernya Payment dan ada file
                Tables\Columns\TextColumn::make('proof')
                    ->label('Bukti')
                    ->getStateUsing(function ($record) {
                        if ($record->source_type === Payment::class && $record->source) {
                            /** @var \App\Models\Payment $payment */
                            $payment = $record->source;
                            if ($payment->proof_file) {
                                return 'Lihat';
                            }
                        }
                        return '-';
                    })
                    ->url(function ($record) {
                        if ($record->source_type === Payment::class && $record->source?->proof_file) {
                            return Storage::url($record->source->proof_file);
                        }
                        return null;
                    }, shouldOpenInNewTab: true)
                    ->openUrlInNewTab()
                    ->extraAttributes(['class' => 'text-primary-600 underline'])
                    ->toggleable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('type')
                    ->label('Tipe')
                    ->options([
                        'credit' => 'Credit (Masuk)',
                        'debit'  => 'Debit (Keluar)',
                    ]),
                Tables\Filters\Filter::make('tanggal')
                    ->form([
                        Forms\Components\DatePicker::make('from')->label('Dari'),
                        Forms\Components\DatePicker::make('until')->label('Sampai'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when($data['from'] ?? null, fn ($q, $from) => $q->whereDate('transaction_date', '>=', $from))
                            ->when($data['until'] ?? null, fn ($q, $until) => $q->whereDate('transaction_date', '<=', $until));
                    }),
                // Filter khusus Payment Type (DP/Pelunasan/Denda) via morph
                Tables\Filters\SelectFilter::make('payment_type')
                    ->label('Jenis Pembayaran')
                    ->options([
                        'DP'        => 'DP',
                        'Pelunasan' => 'Pelunasan',
                        'Denda'     => 'Denda',
                    ])
                    ->query(function ($query, $state) {
                        if (! $state) return $query;
                        return $query->whereHasMorph(
                            'source',
                            [Payment::class],
                            fn ($q) => $q->where('payment_type', $state)
                        );
                    }),
            ])
            ->actions([
                // Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])->defaultSort('transaction_date', 'desc');
    }
    
    public static function getPages(): array
    {
        return [
            'index' => Pages\ListFinancialReports::route('/'),
            'create' => Pages\CreateFinancialReport::route('/create'),
            'edit' => Pages\EditFinancialReport::route('/{record}/edit'),
        ];
    }
}
