<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CustomerResource\Pages;
use App\Filament\Resources\CustomerResource\RelationManagers;
use App\Models\Customer;
use Filament\Forms;
use Filament\Forms\Get;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Radio;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\BadgeColumn;
use Illuminate\Validation\Rules\Enum;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;

class CustomerResource extends Resource
{
    protected static ?string $model = Customer::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';
    protected static ?string $navigationLabel = 'Pelanggan';
    protected static ?string $modelLabel = 'Customer';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Data Customer')
                    ->columns(2)
                    ->schema([
                        TextInput::make('name')
                            ->label('Nama')
                            ->required(),

                        TextInput::make('nik')
                            ->label('No. KTP')
                            ->required()
                            ->maxLength(16)->minLength(16)->rules(['digits:16']),
                        
                        TextInput::make('phone')
                            ->label('Nomor HP')
                            ->tel()
                            ->required(),

                        TextInput::make('emergency_number')
                            ->tel()
                            ->required(),

                        Textarea::make('address')
                            ->label('Alamat')
                            ->rows(2),

                        Radio::make('gender')
                            ->label('Jenis Kelamin')
                            ->options([
                                'male' => 'Laki-laki',
                                'female' => 'Perempuan',
                            ])
                            ->inline()
                            ->required(),

                        DatePicker::make('birth_date')
                            ->label('Tanggal Lahir')
                            ->required(),

                        Select::make('category')
                            ->label('Kategori Customer')
                            ->options([
                                'karyawan' => 'Karyawan',
                                'mahasiswa' => 'Mahasiswa',
                                'wisatawan' => 'Wisatawan',
                            ])
                            ->reactive()
                            ->required(),
                    ]),

                Section::make('Dokumen Pendukung')
                    ->schema([
                        // Semua kategori butuh KTP
                        SpatieMediaLibraryFileUpload::make('ktp')
                            ->label('Foto KTP')
                            ->collection('ktp')
                            ->image()
                            ->imageEditor()
                            ->required(),
                
                        SpatieMediaLibraryFileUpload::make('sim')
                            ->label('Foto SIM (Opsional)')
                            ->collection('sim')
                            ->image()
                            ->imageEditor()
                            ->preserveFilenames()
                            ->columnSpanFull(),
                
                        // Sosial Media
                        TextInput::make('social_media_link')
                            ->label('Akun Sosmed')
                            ->placeholder('@instagram / link')
                            ->columnSpanFull()
                            ->required(),
                
                        // Wisatawan
                        SpatieMediaLibraryFileUpload::make('travel_ticket')
                            ->label('Tiket Perjalanan')
                            ->collection('travel_ticket')
                            ->image()
                            ->visible(fn ($get) => $get('category') === 'wisatawan')
                            ->requiredIf('category', 'wisatawan'),
                
                        SpatieMediaLibraryFileUpload::make('hotel_ticket')
                            ->label('Tiket Penginapan')
                            ->collection('hotel_ticket')
                            ->image()
                            ->visible(fn ($get) => $get('category') === 'wisatawan')
                            ->requiredIf('category', 'wisatawan'),
                
                        // Mahasiswa
                        SpatieMediaLibraryFileUpload::make('ktm')
                            ->label('KTM + Bukti Aktif')
                            ->collection('ktm')
                            ->image()
                            ->visible(fn ($get) => $get('category') === 'mahasiswa')
                            ->requiredIf('category', 'mahasiswa'),
                
                        SpatieMediaLibraryFileUpload::make('kos')
                            ->label('Bukti Kos / Tempat Tinggal')
                            ->collection('kos')
                            ->image()
                            ->visible(fn ($get) => in_array($get('category'), ['mahasiswa', 'karyawan']))
                            ->requiredIf('category', 'mahasiswa'),
                
                        // Karyawan
                        SpatieMediaLibraryFileUpload::make('id_card')
                            ->label('ID Kerja')
                            ->collection('id_card')
                            ->image()
                            ->visible(fn ($get) => $get('category') === 'karyawan')
                            ->requiredIf('category', 'karyawan'),
                
                        SpatieMediaLibraryFileUpload::make('salary_slip')
                            ->label('Slip Gaji')
                            ->collection('salary_slip')
                            ->image()
                            ->visible(fn ($get) => $get('category') === 'karyawan')
                            ->requiredIf('category', 'karyawan'),
                
                        SpatieMediaLibraryFileUpload::make('coworker_doc')
                            ->label('Data Rekan Kerja (KTP / ID Kerja)')
                            ->collection('coworker_doc')
                            ->image()
                            ->visible(fn ($get) => $get('category') === 'karyawan')
                            ->requiredIf('category', 'karyawan'),
                
                        TextInput::make('colleague_info')
                            ->label('Nomor WA Rekan Kerja')
                            ->tel()
                            ->visible(fn ($get) => $get('category') === 'karyawan')
                            ->requiredIf('category', 'karyawan'),
                    ])
                    ->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')->searchable()->sortable(),
                Tables\Columns\TextColumn::make('nik')->searchable(),
                Tables\Columns\TextColumn::make('phone'),
                Tables\Columns\TextColumn::make('category')->badge(),
                Tables\Columns\TextColumn::make('status')->badge(),
                Tables\Columns\TextColumn::make('social_media_link')->label('Sosmed')->limit(30),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                Tables\Filters\SelectFilter::make('category')
                    ->options([
                        'wisatawan' => 'Wisatawan',
                        'mahasiswa' => 'Mahasiswa',
                        'karyawan' => 'Karyawan',
                    ]),
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'active' => 'Aktif',
                        'blacklisted' => 'Blacklist',
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCustomers::route('/'),
            'create' => Pages\CreateCustomer::route('/create'),
            'edit' => Pages\EditCustomer::route('/{record}/edit'),
        ];
    }
}
