<?php

namespace App\Filament\Pages;

use App\Models\Motorcycle;
use App\Models\Rental;
use App\Models\Customer;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Pages\Page;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Carbon;

class AvailableMotorcycles extends Page implements Tables\Contracts\HasTable, Forms\Contracts\HasForms
{
    use Tables\Concerns\InteractsWithTable;
    use Forms\Concerns\InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-magnifying-glass';
    protected static string $view = 'filament.pages.available-motorcycles';
    protected static ?string $title = 'Cek Motor Tersedia';
    protected static ?string $navigationGroup = 'Rental Motor';

    public ?string $startDate = null;
    public ?string $endDate = null;

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make(2)->schema([
                    Forms\Components\DateTimePicker::make('startDate')
                        ->label('Tanggal & Jam Mulai')
                        ->seconds(false)
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->resetTable()),
                    Forms\Components\DateTimePicker::make('endDate')
                        ->label('Tanggal & Jam Selesai')
                        ->seconds(false)
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->resetTable()),
                ]),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(Motorcycle::query())
            ->columns([
                Tables\Columns\TextColumn::make('plate_number')->label('No. Polisi'),
                Tables\Columns\TextColumn::make('brand')->label('Merk'),
                Tables\Columns\TextColumn::make('type')->label('Tipe'),
                Tables\Columns\TextColumn::make('daily_price')
                    ->label('Harga / Hari')
                    ->money('IDR', true),
                Tables\Columns\BadgeColumn::make('status_booking')
                    ->label('Status')
                    ->getStateUsing(function ($record) {
                        if (!$this->startDate || !$this->endDate) {
                            return '-';
                        }

                        $start = Carbon::parse($this->startDate);
                        $end = Carbon::parse($this->endDate);

                        $hasBooking = $record->rentals()
                            ->where(function ($q) use ($start, $end) {
                                $q->whereBetween('start_date', [$start, $end])
                                    ->orWhereBetween('end_date', [$start, $end])
                                    ->orWhere(function ($nested) use ($start, $end) {
                                        $nested->where('start_date', '<=', $start)
                                            ->where('end_date', '>=', $end);
                                    });
                            })
                            ->whereNotIn('status', ['cancelled', 'completed'])
                            ->exists();

                        return $hasBooking ? 'Sudah Dibooking' : 'Tersedia';
                    })
                    ->colors([
                        'danger' => 'Sudah Dibooking',
                        'success' => 'Tersedia',
                    ]),
            ])
            ->actions([
                Action::make('Sewa')
                    ->label('Sewa')
                    ->color('primary')
                    ->visible(fn ($record) => true)
                    ->form([
                        Forms\Components\Select::make('customer_id')
                            ->label('Pilih Customer')
                            ->options(Customer::pluck('name', 'id'))
                            ->searchable()
                            ->required(),
                        Forms\Components\DateTimePicker::make('start_date')
                            ->label('Tanggal & Jam Mulai')
                            ->seconds(false)
                            ->required()
                            ->default(fn () => $this->startDate),
                        Forms\Components\DateTimePicker::make('end_date')
                            ->label('Tanggal & Jam Selesai')
                            ->seconds(false)
                            ->required()
                            ->default(fn () => $this->endDate),
                    ])
                    ->action(function ($record, array $data) {
                        $start = Carbon::parse($data['start_date']);
                        $end = Carbon::parse($data['end_date']);

                        $days = max(1, ceil($start->floatDiffInHours($end) / 24));

                        Rental::create([
                            'rental_number' => 'RENT-' . strtoupper(uniqid()),
                            'customer_id' => $data['customer_id'],
                            'motorcycle_id' => $record->id,
                            'start_date' => $start,
                            'end_date' => $end,
                            'price_per_day' => $record->daily_price,
                            'total_price' => $record->daily_price * $days,
                            'status' => 'booked',
                        ]);

                        Notification::make()
                            ->title('Motor berhasil disewa!')
                            ->success()
                            ->send();
                    }),
            ]);
    }
}