<?php

namespace App\Filament\Resources;

use App\Filament\Resources\BookResource\Pages;
use App\Filament\Resources\BookResource\RelationManagers;
use App\Models\Book;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Components\FileUpload;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\{TextInput, Textarea, Select, MultiSelect};

class BookResource extends Resource
{
    protected static ?string $model = Book::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Katalog Buku';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('title')
                    ->required()
                    ->maxLength(255),

                TextInput::make('isbn')
                    ->label('ISBN')
                    ->maxLength(20)
                    ->unique(ignoreRecord: true),

                TextInput::make('publication_year')
                    ->numeric()
                    ->minValue(1000)
                    ->maxValue(date('Y')),

                TextInput::make('edition')
                    ->maxLength(100),

                Textarea::make('description')
                    ->rows(4)
                    ->maxLength(1000),

                Select::make('publisher_id')
                    ->label('Publisher')
                    ->relationship('publisher', 'name')
                    ->searchable()
                    ->preload(),

                Select::make('location_id')
                    ->label('Location')
                    ->relationship('location', 'code')
                    ->searchable()
                    ->preload(),

                MultiSelect::make('authors')
                    ->relationship('authors', 'name')
                    ->label('Authors')
                    ->required()
                    ->preload()
                    ->searchable(),

                MultiSelect::make('categories')
                    ->relationship('categories', 'name')
                    ->label('Categories')
                    ->required()
                    ->preload()
                    ->searchable(),

                FileUpload::make('cover_image')
                    ->label('Cover Buku')
                    ->image()
                    ->imageEditor()
                    ->directory('book-covers')
                    ->maxSize(2048) // 2 MB
                    ->nullable(),

                Select::make('status')
                    ->options([
                        'Available' => 'Available',
                        'Borrowed' => 'Borrowed',
                        'Reserved' => 'Reserved',
                    ])
                    ->default('Available'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title')->searchable()->sortable(),
                TextColumn::make('isbn')->label('ISBN'),
                TextColumn::make('publisher.name')->label('Publisher'),
                TextColumn::make('location.code')->label('Location'),
                TextColumn::make('status')->badge()->color(fn (string $state) => match($state) {
                    'Available' => 'success',
                    'Borrowed' => 'danger',
                    'Reserved' => 'warning',
                }),
                TextColumn::make('publication_year')->label('Year'),
                ImageColumn::make('cover_image')
                    ->label('Cover')
                    ->disk('public') // jika disimpan di storage/app/public
                    ->width(50)
                    ->height(70),
            ])->defaultSort('title')
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBooks::route('/'),
            'create' => Pages\CreateBook::route('/create'),
            'edit' => Pages\EditBook::route('/{record}/edit'),
        ];
    }

    public static function getHeaderWidgets(): array
    {
        return [
            \App\Filament\Widgets\TotalBooksStats::class,
        ];
    }
}
