<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TransactionResource\Pages;
use App\Filament\Resources\TransactionResource\RelationManagers;
use App\Models\Transaction;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Carbon\Carbon;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use Filament\Tables\Filters\Filter;

class TransactionResource extends Resource
{
    protected static ?string $model = Transaction::class;

    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';
    protected static ?string $navigationLabel = 'Transaksi';
    protected static ?string $modelLabel = 'Data Transaksi';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('type')
                    ->options([
                        'cash_in' => 'Cash In (Pemasukan)',
                        'cash_out' => 'Cash Out (Pengeluaran)',
                    ])
                    ->required(),

                Select::make('category')
                    ->options([
                        'pawn_out' => 'Pencairan Gadai',
                        'redeem_in' => 'Pembayaran Tebusan',
                        'jasa' => 'Jasa Gadai',
                        'denda' => 'Denda Keterlambatan',
                        'admin' => 'Biaya Administrasi',
                    ])
                    ->required(),

                Select::make('gadget_id')
                    ->relationship('gadget', 'model')
                    ->searchable()
                    ->preload()
                    ->label('Gadget'),

                Select::make('user_id')
                    ->relationship('user', 'name')
                    ->searchable()
                    ->preload()
                    ->label('Petugas'),

                DatePicker::make('due_date')
                    ->label('Tanggal Jatuh Tempo'),

                TextInput::make('amount')
                    ->label('Jumlah (Rp)')
                    ->numeric()
                    ->required(),
                
                Textinput::make('nomor_nota')
                    ->label('No. Nota')
                    ->numeric()
                    ->nullable(),

                Textarea::make('notes')
                    ->label('Catatan'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('created_at')->label('Tanggal')->dateTime()->sortable(),
                TextColumn::make('type')->label('Tipe')->badge()->color(fn($state) => $state === 'cash_in' ? 'success' : 'danger'),
                TextColumn::make('category')->label('Kategori')->sortable(),
                TextColumn::make('gadget.model')->label('HP')->sortable()->searchable(),
                TextColumn::make('user.name')->label('Petugas'),
                TextColumn::make('amount')->label('Jumlah Transaksi')->money('IDR', locale: 'id'),
                TextColumn::make('nomor_nota')->label('No. Nota')->sortable()->searchable(),
                TextColumn::make('due_date')->label('Jatuh Tempo')->date(),
            ])
            ->filters([
                Filter::make('date_range')
                ->form([
                    DatePicker::make('from')->label('Dari Tanggal'),
                    DatePicker::make('until')->label('Sampai Tanggal'),
                ])
                ->query(function ($query, array $data) {
                    return $query
                        ->when($data['from'], fn ($q) => $q->whereDate('created_at', '>=', $data['from']))
                        ->when($data['until'], fn ($q) => $q->whereDate('created_at', '<=', $data['until']));
                }),
            ])
            ->actions([
                Tables\Actions\CreateAction::make()->hidden(fn () => auth()->user()->role !== 'admin'),
                Action::make('isi_nomor_nota')
                    ->label('Isi Nomor Nota')
                    ->form([
                        Forms\Components\TextInput::make('nomor_nota')
                            ->label('Nomor Nota')
                            ->required(),
                    ])
                    ->action(function ($record, array $data) { 
                        $record->update([
                            'nomor_nota' => $data['nomor_nota'],
                        ]);
                    })
                    ->visible(fn ($record) => blank($record->nomor_nota)) // hanya tampil jika belum ada
                    ->icon('heroicon-o-document-text')
                    ->modalHeading('Input Nomor Nota'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])->defaultSort('created_at', 'desc');
    }
    
    public static function canCreate(): bool
    {
        return false;
    }
    
    // public static function canEdit(): bool
    // {
    //     return false;
    // }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }
    
    public static function canViewAny(): bool
    {
        return auth()->user()->hasRole('admin') || auth()->user()->hasRole('cashier');
    }
    
    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()->hasRole('admin') || auth()->user()->hasRole('cashier');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTransactions::route('/'),
            'create' => Pages\CreateTransaction::route('/create'),
            'edit' => Pages\EditTransaction::route('/{record}/edit'),
        ];
    }
}
