<?php

namespace App\Filament\Resources;

use App\Filament\Resources\GadgetResource\Pages;
use App\Filament\Resources\GadgetResource\RelationManagers;
use App\Models\Gadget;
use App\Models\Transaction;
use App\Models\CashTransaction;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Forms\Components\Hidden;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class GadgetResource extends Resource
{
    protected static ?string $model = Gadget::class;

    protected static ?string $navigationIcon = 'heroicon-o-device-phone-mobile';
    protected static ?string $navigationLabel = 'Handphone';
    protected static ?string $modelLabel = 'Data Handphone';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Tabs::make('Gadget Info')
            ->tabs([ 
                Forms\Components\Tabs\Tab::make('Info HP')
                    ->schema([
                        // Hidden::make('customer_id')
                        //     ->default(fn () => request()->get('customer_id')),
                        Forms\Components\Select::make('brand')->label('Brand')->required()
                            ->options(['Apple' => 'Apple','Huawei'=>'Huawei', 'Infinix'=>'Infinix', 'Itel' => 'Itel',
                            'Oppo' => 'Oppo', 'Poco' => 'Poco', 'Realme' => 'Realme', 'Samsung' => 'Samsung', 
	                        'Tecno'=>'Tecno', 'Vivo' => 'Vivo', 'Xiaomi' => 'Xiaomi', 'ZTE'=>'ZTE', 'Nubia'=>'Nubia', 'Sony'=>'Sony', 'Canon'=>'Canon']),
                        Forms\Components\TextInput::make('model')->label('Model')->required(),
                        Forms\Components\TextInput::make('imei')->label('IMEI')->required()
                            ->unique(Gadget::class, 'imei', ignoreRecord: true),
                        Forms\Components\TextInput::make('pawn_value')->label('Nilai Gadai')
                            ->numeric()->prefix('Rp')->extraAttributes(['class'=>'uang'])->required()
                            ->reactive() // penting: supaya bisa trigger perubahan
                            ->afterStateUpdated(function (callable $set, $state) {
                                if ($state == 0) {
                                    $set('admin_fee', 0);
                                } elseif ($state <= 500000 && $state > 0) {
                                    $set('admin_fee', 10000);
                                } elseif ($state <= 999000) {
                                    $set('admin_fee', 15000);
                                } else {
                                    $set('admin_fee', 20000);
                                }

                                // Service fee 10% dari pawn_value
                                $serviceFee = (int) round($state * 0.1);
                                $set('service_fee', $serviceFee);
                            }),
                            //->disabled(fn ($record) => filled($record?->pawn_value)),
                        Forms\Components\TextInput::make('admin_fee')->label('Biaya Admin')
                            ->numeric()->prefix('Rp')->extraAttributes(['class'=>'uang'])->required()
                            ->disabled()->dehydrated(),
                        Forms\Components\TextInput::make('service_fee')->label('Bea Jasa Gadai')->numeric()
                            ->prefix('Rp')->extraAttributes(['class'=>'uang'])->required()
                            ->disabled()->dehydrated(),
                        Forms\Components\DatePicker::make('start_date')->label('Tanggal Gadai')->required()
                            ->default(\Carbon\Carbon::now())->disabled()->dehydrated(),
                        Forms\Components\DatePicker::make('end_date')->label('Jatuh Tempo')
                            ->default(\Carbon\Carbon::now()->addDays(30))->required()->disabled()->dehydrated(),
                        Forms\Components\TextInput::make('hp_ram')->label('RAM')->numeric()->required(),
                        Forms\Components\TextInput::make('hp_rom')->label('Internal')->numeric()->required(),
                        Forms\Components\TextInput::make('nomor_nota')
                            ->label('No Nota')->required()
                            ->maxLength(50)
                            ->disabled(fn ($record) => filled($record?->nomor_nota)),
                        Forms\Components\Select::make('status')
                            ->options(['active' => 'Aktif', 'redeem' => 'Sudah Ditebus', 'sold' => 'Dijual',])
                            ->required(),
                    ])->columns(2),
                
                Forms\Components\Tabs\Tab::make('Cek Fisik')
                    ->schema([
                        Forms\Components\Select::make('cf_case_condition')
                            ->label('Body Casing (Kondisi)')
                            ->options(['mulus'=>'Mulus', 'tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cf_case_quality')
                            ->label('Body Casing (Kualitas)')
                            ->options(['lecet < 3'=>'Lecet < 3', 'lecet 4-6'=>'Lecet 4-6', 'lecet > 7'=>'Lecet > 7',])
                            ->required(),
                        Forms\Components\Select::make('cf_glue_marks')
                            ->label('Bekas Lem')
                            ->options(['ada'=>'Ada', 'tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cf_lcd_status')
                            ->label('LCD')
                            ->options(['ori'=>'Ori', 'gantian'=>'Gantian'])
                            ->required(),
                        Forms\Components\Select::make('cf_lcd_condition')
                            ->label('Kondisi LCD')
                            ->options(['nyala'=>'Nyala', 'mati'=>'Mati'])
                            ->required(),
                        Forms\Components\Select::make('cf_lcd_quality')
                            ->label('Kualitas LCD')
                            ->options(['super'=>'Super', 'baik'=>'Baik', 'cukup'=>'Cukup', 'jelek'=>'Jelek'])
                            ->required(),
                    ])->columns(2),                
                Forms\Components\Tabs\Tab::make('Cek Software')
                    ->schema([
                        Forms\Components\Select::make('cs_battery_condition')
                            ->label('Baterai (Kondisi)')
                            ->options(['Normal'=>'Normal', 'Melembung'=>'Melembung'])
                            ->required(),
                        Forms\Components\Select::make('cs_battery_quality')
                            ->label('Baterai (Kualitas)')
                            ->options(['Normal'=>'Normal', 'Ngedrop'=>'Ngedrop'])
                            ->required(),
                        Forms\Components\Select::make('cs_restart3x_condition')
                            ->label('Restart 3x (Kondisi)')    
                            ->options(['Normal'=>'Normal', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cs_screen_condition')
                            ->label('Layar HP (Kondisi)')
                            ->options(['Normal'=>'Normal', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cs_screen_quality')
                            ->label('Layar HP (Kualitas)')
                            ->options(['Mulus'=>'Mulus', 'Retak'=>'Retak', 'Bergaris'=>'Bergaris', 'Shadow'=>'Shadow'])
                            ->required(),
                        Forms\Components\Select::make('cs_sensor_condition')
                            ->label('Sensor (Kondisi)')
                            ->options(['Normal'=>'Normal', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cs_sound_condition')
                            ->label('Suara (Kondisi)')
                            ->options(['Normal'=>'Normal', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cs_flashlight_condition')
                            ->label('Flash/Senter (Kondisi)')
                            ->options(['Normal'=>'Normal', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('cs_screenfunction_condition')
                            ->label('Fungsi Layar (Kondisi)')
                            ->options(['Bisa Multitouch'=>'Bisa Multitouch', 'Tidak Bisa Multitouch'=>'Tidak Bisa Multitouch'])
                            ->required(),
                        Forms\Components\Select::make('cs_front_cam_condition')
                            ->label('Kamera Depan (Kondisi)')
                            ->options(['Nyala'=>'Nyala', 'Mati'=>'Mati'])
                            ->required(),
                        Forms\Components\Select::make('cs_front_cam_quality')
                            ->label('Kamera Depan (Kualitas)')
                            ->options(['Normal'=>'Normal', 'Blur'=>'Blur'])
                            ->required(),
                        Forms\Components\Select::make('cs_main_cam_condition')
                            ->label('Kamera Belakang (Kondisi)')
                            ->options(['Nyala'=>'Nyala', 'Mati'=>'Mati'])
                            ->required(),
                        Forms\Components\Select::make('cs_main_cam_quality')
                            ->label('Kamera Belakang (Kualitas)')
                            ->options(['Normal'=>'Normal', 'Blur'=>'Blur'])
                            ->required(),
                        Forms\Components\Select::make('cs_signal_condition')
                            ->label('Sinyal (Kondisi)')
                            ->options(['Normal'=>'Normal', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\TextInput::make('imei_box_phone')
                            ->label('Imei Kotak HP')
                            ->required(),
                        Forms\Components\Select::make('charger_condition')
                            ->label('Cas')
                            ->options(['Bisa'=>'Bisa', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('charger_status')
                            ->label('Kondisi Charger')
                            ->options(['Ngisi'=>'Ngisi', 'Tidk'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('box')
                            ->label('Dus Box')   
                            ->options(['Ada'=>'Ada', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('account_check')
                            ->label('Akun Bawaan')
                            ->options(['Ada'=>'Ada', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('completeness')
                            ->label('Kelengkapan') 
                            ->options(['HP+Box+Cas'=>'HP+Box+Cas', 'HP+Box'=>'HP+Box', 'HP+Cas'=>'HP+Cas', 'HP Only'=>'HP Only'])
                            ->required(),
                        Forms\Components\Select::make('signature')
                            ->label('Tanda Tangan')
                            ->options(['Sama'=>'Sama', 'Tidak'=>'Tidak'])
                            ->required(),
                        Forms\Components\Select::make('device_check_playstore')
                            ->label('Device Check')
                            ->options(['Sama'=>'Sama', 'Tidak'=>'Tidak'])
                            ->required(),
                        Hidden::make('user_id')->default(fn () => Auth()?->id()),
                ])->columns(2),
            ])->columnSpanFull(),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nomor_nota')->label('No. Nota')->searchable(),
                Tables\Columns\TextColumn::make('model')->searchable(),
                Tables\Columns\TextColumn::make('pawn_value')->label('Nilai Gadai')->money('IDR'),
                Tables\Columns\TextColumn::make('service_fee')->label('Jasa Gadai')
                    ->getStateUsing(function ($record) {
                        // Jika sudah ditebus, ambil nilai dari database
                        if ($record->status === 'redeem') {
                            return number_format($record->service_fee ?? 0, 0, ',', '.');
                        }
                
                        // Jika belum ditebus, hitung secara dinamis
                        $today = now();
                        $startDate = Carbon::parse($record->start_date);
                        $hariGadai = $startDate->diffInDays($today, false);
                
                        // Hitung jasa gadai
                        $jasa = $hariGadai <= 7
                            ? $record->pawn_value * 0.05
                            : $record->pawn_value * 0.10;
                
                        return number_format($jasa, 0, ',', '.');
                    })
                    ->alignRight()
                    ->sortable(),
                Tables\Columns\TextColumn::make('denda')
                    ->getStateUsing(function ($record) {
                        // Kalau sudah ditebus, ambil nilai tetap dari DB
                        if ($record->status === 'redeem') {
                            return number_format($record->denda ?? 0, 0, ',', '.');
                        }
                
                        // Kalau belum ditebus, hitung denda dinamis
                        $today = now();
                        $due = Carbon::parse($record->end_date);
                
                        // Hitung jumlah hari lewat jatuh tempo
                        $overdueDays = $due->isPast() ? $due->diffInDays($today) : 0;
                        $denda = 0;
                
                        // Denda 5% jika lewat 1–7 hari
                        if ($overdueDays > 0 && $overdueDays <= 7) {
                            $denda = $record->pawn_value * 0.05;
                        }
                        // Denda 10% jika lewat 8–31 hari
                        elseif ($overdueDays > 7 && $overdueDays <= 31) {
                            $denda = $record->pawn_value * 0.10;
                        }
                
                        return number_format($denda, 0, ',', '.');
                    })
                    ->alignRight()
                    ->sortable(),
                Tables\Columns\BadgeColumn::make('status')
                ->label('Jenis')
                ->colors([
                    'redeem' => 'success',
                    'active' => 'danger',
                    'extend' => 'warning'
                ])
                ->formatStateUsing(fn ($state) => match ($state) {
                    'redeem' => 'Sudah Ditebus',
                    'active' => 'Aktif digadai',
                    'extend' => 'Diperpanjang',
                    default => ucfirst($state),
                }),
                Tables\Columns\TextColumn::make('end_date')->label('Jatuh Tempo')->date(),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label('Filter Status')
                    ->options([
                        'active' => 'Aktif',
                        'redeem' => 'Sudah Tebus',
                        'extend' => 'Perpanjangan',
                    ])
                    ->placeholder('Semua Status'),
            ])
            ->actions([
                Tables\Actions\Action::make('redeem')
                    ->label('Tebus')
                    ->icon('heroicon-o-banknotes')
                    ->requiresConfirmation(false)
                    ->visible(fn ($record) => $record->status !== 'redeem')
                    ->form(function ($record) {
                        $today = now();
                        $due = Carbon::parse($record->end_date);
                        $overdueDays = $due->isPast() ? $due->diffInDays($today) : 0;
                        $startDate = Carbon::parse($record->start_date);
                
                        // Hitung denda
                        $denda = 0;
                        if ($overdueDays > 0 && $overdueDays <= 7) {
                            $denda = $record->pawn_value * 0.05;
                        } elseif ($overdueDays > 7 && $overdueDays <= 31) {
                            $denda = $record->pawn_value * 0.10;
                        }
                
                        // Hitung jasa gadai
                        $hariGadai = $today->diffInDays($startDate, false);
                        $jasa = $hariGadai <= 0
                            ? $record->pawn_value * 0.05
                            : $record->pawn_value * 0.10;
                
                        $totalBayar = $record->pawn_value + $jasa + $denda;
                
                        return [
                            \Filament\Forms\Components\ViewField::make('summary')
                                ->view('filament.forms.redeem-summary', [
                                    'record' => $record,
                                    'denda' => $denda,
                                    'jasa' => $jasa,
                                    'totalBayar' => $totalBayar,
                                ]),
                        ];
                    })
                    ->action(function ($record, array $data) {
                        $today = now();
                        $due = \Carbon\Carbon::parse($record->end_date);
                        $overdueDays = $due->isPast() ? $due->diffInDays($today) : 0;
                        $startDate = \Carbon\Carbon::parse($record->start_date);

                        $denda = 0;
                        if($overdueDays > 0 && $overdueDays <= 7) {
                            $denda = $record->pawn_value * 0.05;
                        } elseif ($overdueDays > 7 && $overdueDays <= 31) {
                            $denda = $record->pawn_value * 0.10;
                        }

                         // Hitung jasa gadai
                        $hariGadai = $today->diffInDays($startDate, false);
                        $jasa = 0;

                        if ($hariGadai <= 0) {
                            // Hari yang sama
                            $jasa = $record->pawn_value * 0.05;
                        } else {
                            $jasa = $record->pawn_value * 0.10;
                        }

                        $totalBayar = $record->pawn_value + $record->admin_fee + $jasa + $denda;

                        // update status dan nilai jasa serta denda
                        $record->update([
                            'status' => 'redeem',
                            'service_fee' => $jasa,
                            'denda' => $denda,
                        ]);

                        // simpan transaksi
                        Transaction::create([
                            'nomor_nota' => $record->nomor_nota,
                            'type' => 'cash_in',
                            'category' => 'redeem_in',
                            'amount' => $record->pawn_value,
                            'user_id' => auth()->id(),
                            'gadget_id' => $record->id,
                            'due_date' => $record->end_date,
                            'notes' => 'Penebusan pokok'. $record->brand . ' ' . $record->model,
                        ]);
                        Transaction::create([
                            'nomor_nota' => $record->nomor_nota,
                            'type' => 'cash_in',
                            'category' => 'jasa',
                            'amount' => $jasa,
                            'user_id' => auth()->id(),
                            'gadget_id' => $record->id,
                            'due_date' => $record->end_date,
                            'notes' => 'Jasa gadai'. $record->brand . ' ' . $record->model,
                        ]);
                        Transaction::create([
                            'nomor_nota' => $record->nomor_nota,
                            'type' => 'cash_in',
                            'category' => 'denda',
                            'amount' => $denda,
                            'user_id' => auth()->id(),
                            'gadget_id' => $record->id,
                            'due_date' => $record->end_date,
                            'notes' => 'Denda keterlambatan'. $record->brand . ' ' . $record->model,
                        ]);
                        CashTransaction::create([
                            'date' => now(),
                            'description' => 'Tebus Gadai: ' . $record->brand . ' ' . $record->model,
                            'type' => 'credit', // uang masuk ke kas
                            'amount' => $record->pawn_value, // asumsi kamu punya ini
                            'reference_type' => Gadget::class,
                            'reference_id' => $record->nomor_nota,
                            'created_by' => auth()->id(),
                        ]);
                        CashTransaction::create([
                            'date' => now(),
                            'description' => 'Jasa Gadai: ' . $record->brand . ' ' . $record->model,
                            'type' => 'credit', // uang masuk ke kas
                            'amount' => $jasa, // asumsi kamu punya ini
                            'reference_type' => Gadget::class,
                            'reference_id' => $record->nomor_nota,
                            'created_by' => auth()->id(),
                        ]);
                        CashTransaction::create([
                            'date' => now(),
                            'description' => 'Denda Gadai: ' . $record->brand . ' ' . $record->model,
                            'type' => 'credit', // uang masuk ke kas
                            'amount' => $denda, // asumsi kamu punya ini
                            'reference_type' => Gadget::class,
                            'reference_id' => $record->nomor_nota,
                            'created_by' => auth()->id(),
                        ]);
                    })
                    ->hidden(fn (Gadget $record) => $record->status === 'redeem'),
                Tables\Actions\Action::make('print-nota')
                    ->label('Cetak Nota')
                    ->icon('heroicon-o-printer')
                    ->url(fn (Gadget $record) => route('gadgets.print-nota', $record))
                    ->openUrlInNewTab()
                    ->visible(fn (Gadget $record) => $record->status === 'redeem'),
                Tables\Actions\Action::make('extend')
                    ->label('Perpanjang')
                    ->icon('heroicon-o-arrow-path-rounded-square')
                    ->requiresConfirmation()
                    ->action(function (Model $record): void {
                        if ($record->extension_count >= 3) {
                            Notification::make()
                                ->title('Maksimal perpanjangan hanya 3 kali.')
                                ->danger()
                                ->send();
                            return;
                        }
                
                        // Hitung tanggal baru (misalnya +30 hari)
                        $newEndDate = \Carbon\Carbon::parse($record->end_date)->addDays(30);
                        $newServiceFee = $record->service_fee; // bisa dinaikkan jika perlu
                
                        $record->update([
                            'status' => 'extend',
                            'end_date' => $newEndDate,
                            'service_fee' => $newServiceFee,
                            'extension_count' => $record->extension_count + 1,
                        ]);
                
                        Transaction::create([
                            'nomor_nota' => $record->nomor_nota,
                            'type' => 'cash_in',
                            'category' => 'perpanjangan',
                            'amount' => $newServiceFee,
                            'user_id' => auth()->id(),
                            'gadget_id' => $record->id,
                            'due_date' => $newEndDate,
                            'notes' => 'Perpanjangan masa gadai untuk ' . $record->brand . ' ' . $record->model,
                        ]);
                        CashTransaction::create([
                            'date' => now(),
                            'description' => 'Perpanjangan Gadai: ' . $record->brand . ' ' . $record->model,
                            'type' => 'credit', // uang masuk ke kas
                            'amount' => $record->service_fee, // asumsi kamu punya ini
                            'reference_type' => Gadget::class,
                            'reference_id' => $record->nomor_nota,
                            'created_by' => auth()->id(),
                        ]);

                        Notification::make()
                            ->title('Perpanjangan berhasil.')
                            ->success()
                            ->send();
                    })
                    ->hidden(fn (Gadget $record) => $record->status === 'redeem' || $record->extension_count >= 3),
                Tables\Actions\Action::make('print-nota-perpanjangan')
                    ->label('Cetak Nota Perpanjangan')
                    ->icon('heroicon-o-printer')
                    ->url(fn (Gadget $record) => route('gadgets.print-nota-perpanjangan', $record))
                    ->openUrlInNewTab()
                    ->visible(fn (  $record) => $record->status === 'extend'),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
    
    public static function canCreate(): bool
    {
        return false;
    }
    
    public static function canSearch(): bool
    {
        return true;
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }
    
    public static function canViewAny(): bool
    {
        return auth()->user()->hasRole('admin') || auth()->user()->hasRole('cashier');
    }
    
    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()->hasRole('admin') || auth()->user()->hasRole('cashier');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListGadgets::route('/'),
            'create' => Pages\CreateGadget::route('/create'),
            'edit' => Pages\EditGadget::route('/{record}/edit'),
        ];
    }
}
