<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CustomerResource\Pages;
use App\Filament\Resources\CustomerResource\RelationManagers;
use App\Models\Customer;
use Filament\Tables\Actions\Action;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\FileUpload;
use Filament\Tables\Columns\ImageColumn;
use App\Filament\Resources\CustomerResource\RelationManagers\GadgetsRelationManager;

class CustomerResource extends Resource
{
    protected static ?string $model = Customer::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';
    protected static ?string $navigationLabel = 'Customers';
    protected static ?string $pluralLabel = 'Customers';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')->label('Nama')->required()->maxLength(255),
                Forms\Components\TextInput::make('ktp_no')->label('No. KTP')->required()
                    ->maxLength(16)->minLength(16)->rules(['digits:16']),
                Forms\Components\TextInput::make('phone')->label('No Telepon')->required()->maxLength(14),
                Forms\Components\Textarea::make('address')->label('Alamat'),
                FileUpload::make('ktp_document')->label('KTP Upload')
                    ->image()
                    ->imageEditor()
                    ->directory('customers')
                    ->maxSize('12288')
                    ->columnSpanFull()
                    ->visibility('public'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')->label('Nama')->searchable()->sortable(),
                Tables\Columns\TextColumn::make('ktp_no')->label('No. KTP')->searchable()->sortable(),
                //Tables\Columns\TextColumn::make('email'),
                Tables\Columns\TextColumn::make('phone')->label('Telepon'),
                Tables\Columns\TextColumn::make('address')->label('Alamat'),
                ImageColumn::make('ktp_document')->label('KTP Upload')->circular()->disk('public')
                    ->url(fn ($record) => asset('storage/customers/' . $record->ktp_document))
                    ->openUrlInNewTab(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
                Action::make('download_ktp')
                    ->label('Download KTP')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->url(fn ($record) => asset('storage/' . data_get($record, 'ktp_document')))
                    ->openUrlInNewTab()
                    ->visible(fn ($record) => filled(data_get($record, 'ktp_document'))),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            GadgetsRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCustomers::route('/'),
            'create' => Pages\CreateCustomer::route('/create'),
            'edit' => Pages\EditCustomer::route('/{record}/edit'),
        ];
    }
    
    public static function canViewAny(): bool
    {
        return auth()?->user()?->hasRole('admin') || auth()?->user()?->hasRole('cashier');
    }
    
    public static function shouldRegisterNavigation(): bool
    {
        return auth()?->user()?->hasRole('admin') || auth()?->user()?->hasRole('cashier');
    }
}
