<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CashTransactionResource\Pages;
use App\Filament\Resources\CashTransactionResource\RelationManagers;
use App\Models\CashTransaction;
use App\Exports\CashTransactionExport;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Support\RawJs;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Model;
use Maatwebsite\Excel\Facades\Excel;
use App\Filament\Resources\CashTransactionResource\Widgets\CashBalanceWidget;


class CashTransactionResource extends Resource
{
    protected static ?string $model = CashTransaction::class;
    protected static ?string $navigationIcon = 'heroicon-o-banknotes';
    protected static ?string $navigationGroup = 'Keuangan';

    public static function form(Forms\Form $form): Forms\Form
    {
        return $form
            ->schema([
                Forms\Components\DatePicker::make('date')->required(),
                Forms\Components\TextInput::make('description')->required(),
                Forms\Components\Select::make('type')
                    ->options(['debit' => 'Debit (Keluar)', 'credit' => 'Kredit (Masuk)'])
                    ->required(),
                Forms\Components\TextInput::make('amount')
                    ->numeric()
                    ->required(),
            ]);
    }

    public static function table(Tables\Table $table): Tables\Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('date')->date(),
                Tables\Columns\TextColumn::make('description')->limit(30),
                Tables\Columns\BadgeColumn::make('type')
                ->label('Jenis')
                ->colors([
                    'debit' => 'danger',
                    'credit' => 'success',
                ])
                ->formatStateUsing(fn ($state) => match ($state) {
                    'debit' => 'Uang Keluar',
                    'credit' => 'Uang Masuk',
                    default => ucfirst($state),
                }),
                Tables\Columns\TextColumn::make('amount')->money('IDR', true),
                Tables\Columns\TextColumn::make('creator.name')->label('User'),
            ])
            ->defaultSort('date', 'desc')
            ->filters([
                //
            ])
            ->headerActions([
                Action::make('Export Excel')
                    ->icon('heroicon-m-arrow-down-tray')
                    ->action(function () {
                        return Excel::download(new CashTransactionExport, 'cash_transactions.xlsx');
                    }),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCashTransactions::route('/'),
            'create' => Pages\CreateCashTransaction::route('/create'),
            'edit' => Pages\EditCashTransaction::route('/{record}/edit'),
        ];
    }

    public static function getHeaderWidgets(): array
    {
        return [
            CashBalanceWidget::class,
        ];
    }
    
    public static function canViewAny(): bool
    {
        return true; // semua role boleh lihat
    }

    public static function canCreate(): bool
    {
        return !auth()->user()->hasRole('cashier');
    }

    public static function canEdit(Model $record): bool
    {
        return !auth()->user()->hasRole('cashier');
    }

    public static function canDelete(Model $record): bool
    {
        return !auth()->user()->hasRole('cashier');
    }

    public static function canForceDelete(Model $record): bool
    {
        return false;
    }

    public static function canRestore(Model $record): bool
    {
        return false;
    }

    // public static function canViewAny(): bool
    // {
    //     return auth()?->user()?->hasRole('admin') || auth()?->user()?->hasRole('cashier');
    // }
    
    public static function shouldRegisterNavigation(): bool
    {
        return auth()?->user()?->hasRole('admin') || auth()?->user()?->hasRole('cashier');
    }
}
