<?php

namespace App\Filament\Pages;

use App\Models\Transaction;
use App\Models\CashTransaction;
use Illuminate\Support\Carbon;
use App\Exports\FinancialReportExport;
use Maatwebsite\Excel\Facades\Excel;
use Filament\Forms;
use Filament\Pages\Page;
use Filament\Pages\PageWithForm;
use Filament\Forms\Form;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Actions\Action;


class FinancialReport extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-chart-bar';
    protected static string $view = 'filament.pages.financial-report';
    protected static ?string $navigationLabel = 'Laporan Keuangan';
    protected static ?string $navigationGroup = 'Keuangan';

    public string $range = 'daily';
    public ?string $start_date = null;
    public ?string $end_date = null;

    public function getViewData(): array
    {
        $start = now()->startOfDay();
        $end = now()->endOfDay();

        if ($this->range === 'weekly') {
            $start = now()->startOfWeek();
            $end = now()->endOfWeek();
        } elseif ($this->range === 'monthly') {
            $start = now()->startOfMonth();
            $end = now()->endOfMonth();
        } elseif ($this->range === 'custom' && $this->start_date && $this->end_date) {
            $start = Carbon::parse($this->start_date)->startOfDay();
            $end = Carbon::parse($this->end_date)->endOfDay();
        }

        $transactions = Transaction::whereBetween('created_at', [$start, $end])->get();
        $cash = CashTransaction::whereBetween('date', [$start, $end])->get();

        return [
            'income' => $transactions->where('type', 'cash_in')->sum('amount'),
            'expense' => $transactions->where('type', 'cash_out')->sum('amount'),
            'kas_masuk' => $cash->where('type', 'credit')->sum('amount'),
            'kas_keluar' => $cash->where('type', 'debit')->sum('amount'),
            'kas_saldo' => $cash->where('type', 'credit')->sum('amount') - $cash->where('type', 'debit')->sum('amount'),
            'filter' => $this->range,
            'admin_fee' => $transactions->where('category', 'admin')->sum('amount'),
            'jasa' => $transactions->where('category', 'jasa')->sum('amount'),
            'denda' => $transactions->where('category', 'denda')->sum('amount'),
            'perpanjangan' => $transactions->where('category', 'perpanjangan')->sum('amount'),
        ];
    }

    public function exportExcel()
    {
        $start = now()->startOfMonth()->format('Y-m-d');
        $end   = now()->endOfMonth()->format('Y-m-d');

        return Excel::download(new FinancialReportExport($start, $end), 'laporan-keuangan.xlsx');
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('range')
                    ->label('Rentang Waktu')
                    ->options([
                        'daily' => 'Harian',
                        'weekly' => 'Mingguan',
                        'monthly' => 'Bulanan',
                        'custom' => 'Custom',
                    ])
                    ->default('daily')
                    ->reactive()
                    ->afterStateUpdated(fn ($state, callable $set) => $state !== 'custom' ? $set('start_date', null) : null),

                DatePicker::make('start_date')->label('Dari'),
                DatePicker::make('end_date')->label('Sampai'),
            ]);
    }

    public function getHeaderActions(): array
    {
        return [
            Action::make('export_excel')
                ->label('Export Excel')
                ->icon('heroicon-o-arrow-down-tray')
                ->action(function () {
                    $start = $this->form->getState()['start_date'];
                    $end = $this->form->getState()['end_date'];

                    return \Maatwebsite\Excel\Facades\Excel::download(
                        new \App\Exports\FinancialReportExport($start, $end),
                        'laporan-keuangan.xlsx'
                    );
                }),
        ];
    }

    public function applyFilters()
    {
        // Biarkan kosong kalau cuma trigger update
        // Atau bisa logging untuk debug
        logger('Filters applied: ' . $this->range . ', ' . $this->start_date . ' - ' . $this->end_date);
    }
}